// Generic IO functions
#include "io.h"

//          RA4 LED
//          RB6 SW1
//          RB7 SW2
//          RC9 SW3
//          RA3 SW4
//          RB4 SW5
//          RB10 SW6
//          RB11 SW7
//          RA2 REG1EN (5V for audio)
//          RP2 as MCLK
//          SPI2 for I2S on CS4334 DAC RPOR using Pin2 SCK, Pin6 SDO, pin7 CS
//          SPI1 for SD Card (CD on RB13, pin24)
//          RB5 IC3 (amplifier) enable

void LEDSet(unsigned char state){
    if(state){
        LATAbits.LATA4=1;       //on
    }else{
        LATAbits.LATA4=0;       //off        
    }
}

void reg1Set(unsigned char state){
    if(state){
        LATAbits.LATA2=1;       //on
    }else{
        LATAbits.LATA2=0;       //off        
    }    
}

void reg1Ramp(uint32_t state){             //ramped start for limited input supplies
    int32_t n=0;
    if(state){
        for(n=0;n<state;n++){
            LATAbits.LATA2=1;       //turn on
            delayUs(n);
            LATAbits.LATA2=0;       //turn off
            delayUs(state-n);
        }
        LATAbits.LATA2=1;       //leave fully on        
    }else{
        LATAbits.LATA2=0;       //off immediately       
    }        
}

void ampSet(unsigned char state){
    if(state){
        LATBbits.LATB5=1;       //on
    }else{
        LATBbits.LATB5=0;       //off        
    }    
}

void IOIdle(void){          //put everything in low power
    __builtin_disable_interrupts();//disable interrupts
    CCP4CON1bits.ON = 0;    // turn off PWM for MCLK
    SPI2CONbits.ON=0;       //turn off SPI2 (I2S)            
    AD1CON1bits.ON=0;       //shut down if it has been turned on
    //turn off pull-ups? only on sw4 if used as speed control, otherwise needed for trigger to come out of sleep

}

void DeinitSPI(){           //separate function as it gets inited by card libs
    SPI1CONbits.ON=0;       //turn off SPI1 (MicroSD Card)
}

void IOInit(void){      
    ANSELA=0;               //not using any ADC (if needed, turned on later)
    ANSELB=0;
    ANSELC=0;
    AD1CON1bits.ON=0;
    
    TRISAbits.TRISA4=0;     //LED output (need to disable SOSC)
    LATAbits.LATA4=0;       //LED off        
    TRISAbits.TRISA2=0;     //REG1EN (needs POSCMOD off)
    LATAbits.LATA2=0;       //off
    TRISBbits.TRISB5=0;     //IC3 (amp)
    LATBbits.LATB5=0;       //off    
    TRISBbits.TRISB6=1;     //SW1 input 
    CNPUBbits.CNPUB6=1;     //SW1 pullup
    TRISBbits.TRISB7=1;     //SW2 input 
    CNPUBbits.CNPUB7=1;     //SW2 pullup
    TRISCbits.TRISC9=1;     //SW3 input 
    CNPUCbits.CNPUC9=1;     //SW3 pullup
    TRISAbits.TRISA3=1;     //SW4 input  (needs POSCMOD off)
    CNPUAbits.CNPUA3=1;     //SW4 pullup
    TRISBbits.TRISB4=1;     //SW5 input  (disable SOSC)
    CNPUBbits.CNPUB4=1;     //SW5 pullup
    TRISBbits.TRISB10=1;     //SW6 input 
    CNPUBbits.CNPUB10=1;     //SW6 pullup
    TRISBbits.TRISB11=1;     //SW7 input 
    CNPUBbits.CNPUB11=1;     //SW7 pullup        
    TRISBbits.TRISB13=1;     //CD input 
    CNPUBbits.CNPUB13=1;     //CD pullup        
    
    //RPOR allocated pins
    __builtin_disable_interrupts();//disable interrupts
    SYSKEY = 0;                 // force lock
    SYSKEY = 0xAA996655;        // unlock sequence
    SYSKEY = 0x556699AA;        //  
    RPOR0bits.RP2R=11;          // Pin3 => SCCP4(MCLK for I2S)
    RPOR0bits.RP1R=9;           // Pin2 => SCK2
    RPOR1bits.RP8R=8;           // Pin6 => SDO2
    RPOR2bits.RP9R=10;          // Pin7 => CS2    
    SYSKEY = 0;                 // relock
    
    //MCLK nominally 12MHz (half system clock)
    CCP4CON1 = 0;               // PWM on SCCP4
    CCP4CON1bits.CLKSEL=0;      // system clock
    CCP4CON1bits.MOD = 0b0101;  // dual edge compare, buffered
    CCP4CON1bits.TMRPS = 0;     // prescale=1
    CCP4PR=1;                   // period
    CCP4RA=0;                   // rising edge
    CCP4RB=1;                   // falling edge
    CCP4CON1bits.ON = 1;        // turn on PWM, 12MHz
    
    //I2S on SPI2
    SPI2CONbits.ON=0;           //turn off SPI, set PBCLK
    SPI2CONbits.DISSDI=1;       //SDI not used
    SPI2BUF;                    //read rx buffer
    SPI2CON2bits.AUDEN=1;       //enable audio
    SPI2CON2bits.AUDMONO=1;     //mono
    SPI2CON2bits.AUDMOD=0;      //audio mode (I2S)
    SPI2CON2bits.SPISGNEXT=0;   //not sign extended
    SPI2CON2bits.FRMERREN=0;    //ignore frame errors
    SPI2CON2bits.SPIROVEN=0;    //ignore receive overflow
    SPI2CON2bits.SPITUREN=0;    //ignore transmit underrun
    SPI2CON2bits.IGNROV=1;      //ignore receive overflow
    SPI2CON2bits.IGNTUR=1;      //ignore transmit underrun
    SPI2CONbits.MODE32=0;       //16bit data, 32 bit frame    
    SPI2CONbits.MODE16=0;
    SPI2CONbits.MSTEN=1;        //master
    SPI2CONbits.FRMPOL=0;       //for I2S mode
    SPI2CONbits.CKP=1;          //clock polarity
    SPI2BRG=7;                  //bitrate based on peripheral bus clock, 1/16 of MCLK (1/2(n+1))
    
    IEC0=0;                     //disable all individual interrupt controls
    IEC1=0;
    IEC2=0;
    IEC3=0;

    IPC11bits.SPI2TXIP = 3;     //set interrupt priority
    IFS1bits.SPI2TXIF=0;        //clear flag
    IEC1bits.SPI2TXIE=0;        //disable TX interrupt
    __builtin_enable_interrupts();//enable interrupts
    
    SPI2CONbits.ON=1;           //turn on SPI2    
}

unsigned char getSW(unsigned char n){
    switch(n){
        case 1: return PORTBbits.RB6;
        case 2: return PORTBbits.RB7;
        case 3: return PORTCbits.RC9;
        case 4: return PORTAbits.RA3;
        case 5: return PORTBbits.RB4;
        case 6: return PORTBbits.RB10;
        case 7: return PORTBbits.RB11;        
        default: return 0;
    }
}

uint32_t getAnalog(){                       //hardwired to RA3/AN6/SW4/pin 10- working, returns 0-4095
    uint32_t retVal=0;
    //static char analogInit=0;               //do init here if needed, possibly use AD1CON1bits.ON instead
    if(AD1CON1bits.ON!=1){
        CNPUAbits.CNPUA3=1;                 //leave SW4 pullup on to stabilise
        ANSELAbits.ANSA3=1;
        TRISAbits.TRISA3=1;
        AD1CHSbits.CH0SA=6;                 //set AN6 as source
        AD1CHSbits.CH0NA=0;                 //Negative reference is VREFL
        AD1CON3bits.ADRC=1;                 //PBCLK as source for PIC32MMxx, TAD=280ns minimum
        AD1CON3bits.ADCS=8;                 //8=>TAD=16TSRC => TAD=666nS
        AD1CON3bits.SAMC=5;                 //T = 5TAD (does this need to be 14+?)
        AD1CON1bits.ASAM=1;                 //automatic
        AD1CON1bits.SSRC=7;                 //set trigger (auto-convert)
        AD1CON1bits.MODE12=1;               //12 bit mode on
        AD1CON1bits.FORM=0;                 //right justified 12 bit unsigned
        AD1CON2bits.VCFG=0;                 //voltage Reference AVDD/AVSS
        AD1CON2bits.SMPI=0;                 //use 1st buffer postion only
        AD1CON1bits.ON=1;                   //turn on ADC
        AD1CON1bits.SAMP=1;                 //start (others have SAMP=0 to start?)
        
        //analogInit=1;                       //init done
    }
    while(IFS1bits.AD1IF == 0){}            //wait if necessary
    retVal=ADC1BUF0;                        //get result
    IFS1CLR = _IFS1_AD1IF_MASK;             //clear flag
    return retVal;
}